<?php
/**
 * Arena 7x7 - Sistema de Autenticacao
 * Login seguro usando tabela do rAthena
 */

if (!defined('ARENA7X7')) {
    die('Acesso negado.');
}

class Auth {
    
    /**
     * Inicia sessao segura
     */
    public static function startSession() {
        if (session_status() === PHP_SESSION_NONE) {
            ini_set('session.cookie_httponly', 1);
            ini_set('session.use_only_cookies', 1);
            ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
            session_name(SESSION_NAME);
            session_start();
        }
        
        // Regenerar ID da sessao periodicamente
        if (!isset($_SESSION['created'])) {
            $_SESSION['created'] = time();
        } else if (time() - $_SESSION['created'] > 1800) {
            session_regenerate_id(true);
            $_SESSION['created'] = time();
        }
    }
    
    /**
     * Verifica se usuario esta logado
     */
    public static function isLoggedIn() {
        return isset($_SESSION['user_id']) && isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }
    
    /**
     * Retorna dados do usuario logado
     */
    public static function getUser() {
        if (!self::isLoggedIn()) {
            return null;
        }
        return [
            'account_id' => $_SESSION['user_id'],
            'userid' => $_SESSION['userid'],
            'email' => $_SESSION['email'] ?? ''
        ];
    }
    
    /**
     * Retorna account_id do usuario logado
     */
    public static function getUserId() {
        return self::isLoggedIn() ? $_SESSION['user_id'] : null;
    }
    
    /**
     * Tenta fazer login
     */
    public static function login($username, $password) {
        $username = trim($username);
        
        if (empty($username) || empty($password)) {
            return ['success' => false, 'message' => 'Preencha todos os campos.'];
        }
        
        // Buscar usuario na tabela login do rAthena
        $sql = "SELECT `account_id`, `userid`, `user_pass`, `email`, `state` 
                FROM `" . TABLE_LOGIN . "` 
                WHERE `userid` = :username 
                LIMIT 1";
        
        $user = db()->fetchOne($sql, ['username' => $username]);
        
        if (!$user) {
            return ['success' => false, 'message' => 'Usuario ou senha incorretos.'];
        }
        
        // Verificar se conta esta ativa
        if ($user['state'] != 0) {
            return ['success' => false, 'message' => 'Conta bloqueada ou banida.'];
        }
        
        // Verificar senha (rAthena usa MD5 por padrao)
        // Nota: Alguns servidores podem usar outros metodos
        $password_hash = $user['user_pass'];
        
        // Tentar MD5 primeiro (padrao rAthena)
        if ($password_hash === md5($password)) {
            // Senha correta
        }
        // Tentar comparacao direta (caso seja plaintext)
        else if ($password_hash === $password) {
            // Senha correta (plaintext - nao recomendado)
        }
        // Tentar password_verify para bcrypt
        else if (password_verify($password, $password_hash)) {
            // Senha correta (bcrypt)
        }
        else {
            return ['success' => false, 'message' => 'Usuario ou senha incorretos.'];
        }
        
        // Login bem sucedido - criar sessao
        session_regenerate_id(true);
        $_SESSION['user_id'] = $user['account_id'];
        $_SESSION['userid'] = $user['userid'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['logged_in'] = true;
        $_SESSION['login_time'] = time();
        $_SESSION['ip'] = $_SERVER['REMOTE_ADDR'];
        
        return ['success' => true, 'message' => 'Login realizado com sucesso!'];
    }
    
    /**
     * Faz logout
     */
    public static function logout() {
        $_SESSION = [];
        
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        
        session_destroy();
    }
    
    /**
     * Gera token CSRF
     */
    public static function generateCsrfToken() {
        if (empty($_SESSION[CSRF_TOKEN_NAME])) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
        }
        return $_SESSION[CSRF_TOKEN_NAME];
    }
    
    /**
     * Valida token CSRF
     */
    public static function validateCsrfToken($token) {
        if (empty($_SESSION[CSRF_TOKEN_NAME]) || empty($token)) {
            return false;
        }
        return hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }
    
    /**
     * Retorna campo hidden com token CSRF
     */
    public static function csrfField() {
        return '<input type="hidden" name="' . CSRF_TOKEN_NAME . '" value="' . self::generateCsrfToken() . '">';
    }
    
    /**
     * Busca personagens da conta logada
     */
    public static function getCharacters() {
        if (!self::isLoggedIn()) {
            return [];
        }
        
        $sql = "SELECT `char_id`, `name`, `class`, `base_level`, `job_level` 
                FROM `" . TABLE_CHAR . "` 
                WHERE `account_id` = :account_id 
                ORDER BY `name` ASC";
        
        return db()->fetchAll($sql, ['account_id' => self::getUserId()]);
    }
    
    /**
     * Verifica se um char_id pertence ao usuario logado
     */
    public static function ownsCharacter($char_id) {
        if (!self::isLoggedIn()) {
            return false;
        }
        
        $sql = "SELECT COUNT(*) FROM `" . TABLE_CHAR . "` 
                WHERE `char_id` = :char_id AND `account_id` = :account_id";
        
        return db()->fetchColumn($sql, [
            'char_id' => $char_id,
            'account_id' => self::getUserId()
        ]) > 0;
    }
}
