<?php
/**
 * Arena 7x7 - Classe de Conexao PDO
 * Conexao segura com o banco de dados usando PDO
 */

if (!defined('ARENA7X7')) {
    die('Acesso negado.');
}

class Database {
    private static $instance = null;
    private $pdo;
    
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];
            
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            // Nao expor detalhes do erro em producao
            if (defined('PRODUCTION_MODE') && PRODUCTION_MODE) {
                error_log('Database Error: ' . $e->getMessage());
                die('Erro interno do servidor. Por favor, tente novamente mais tarde.');
            } else {
                die('Erro de conexao com o banco de dados: ' . $e->getMessage());
            }
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->pdo;
    }
    
    /**
     * Trata erros de forma segura
     */
    private function handleError($e, $context = 'Query') {
        if (defined('PRODUCTION_MODE') && PRODUCTION_MODE) {
            error_log($context . ' Error: ' . $e->getMessage());
            return null;
        } else {
            throw $e;
        }
    }
    
    /**
     * Executa uma query preparada e retorna todos os resultados
     */
    public function fetchAll($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            $this->handleError($e, 'FetchAll');
            return [];
        }
    }
    
    /**
     * Executa uma query preparada e retorna uma linha
     */
    public function fetchOne($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetch();
        } catch (PDOException $e) {
            $this->handleError($e, 'FetchOne');
            return null;
        }
    }
    
    /**
     * Executa uma query preparada e retorna o valor de uma coluna
     */
    public function fetchColumn($sql, $params = [], $column = 0) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchColumn($column);
        } catch (PDOException $e) {
            $this->handleError($e, 'FetchColumn');
            return null;
        }
    }
    
    /**
     * Executa uma query preparada (INSERT, UPDATE, DELETE)
     */
    public function execute($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            $this->handleError($e, 'Execute');
            return false;
        }
    }
    
    /**
     * Retorna o ultimo ID inserido
     */
    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }
    
    /**
     * Conta registros
     */
    public function count($table, $where = '', $params = []) {
        $sql = "SELECT COUNT(*) FROM `{$table}`";
        if ($where) {
            $sql .= " WHERE {$where}";
        }
        return (int) $this->fetchColumn($sql, $params);
    }
}

// Funcao helper para obter conexao
function db() {
    return Database::getInstance();
}
