<?php
/**
 * Arena 7x7 - Funcoes Auxiliares
 * Helpers e utilidades do sistema
 * ADAPTADO para estrutura de tabelas do banco rwc
 */

if (!defined('ARENA7X7')) {
    die('Acesso negado.');
}

// Desabilitar exibicao de erros em producao
if (defined('PRODUCTION_MODE') && PRODUCTION_MODE) {
    error_reporting(0);
    ini_set('display_errors', '0');
}

/**
 * Escapa HTML para prevenir XSS
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Gera URL segura para jogador usando nome ao inves de ID
 */
function playerUrl($char_name) {
    return 'player.php?name=' . urlencode($char_name);
}

/**
 * Gera URL segura para partida usando hash
 */
function matchUrl($match_id) {
    $hash = substr(hash_hmac('sha256', $match_id, URL_HASH_SECRET), 0, 8);
    return 'match.php?id=' . (int)$match_id . '&h=' . $hash;
}

/**
 * Valida hash de partida
 */
function validateMatchHash($match_id, $hash) {
    $expected = substr(hash_hmac('sha256', $match_id, URL_HASH_SECRET), 0, 8);
    return hash_equals($expected, $hash);
}

/**
 * Busca char_id pelo nome do personagem
 */
function getCharIdByName($char_name) {
    $result = db()->fetchOne(
        "SELECT char_id FROM arena7x7_ranking WHERE char_name = :name LIMIT 1",
        ['name' => $char_name]
    );
    return $result ? $result['char_id'] : null;
}

/**
 * Retorna URL base do site
 */
function baseUrl($path = '') {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $baseDir = dirname($_SERVER['SCRIPT_NAME']);
    $baseDir = rtrim($baseDir, '/');
    return $protocol . '://' . $host . $baseDir . '/' . ltrim($path, '/');
}

/**
 * Retorna URL de imagem de skill do Divine Pride
 */
function skillImage($skill_id) {
    return SKILL_IMG_URL . (int)$skill_id . '.png';
}

/**
 * Retorna URL de imagem de item do Divine Pride
 */
function itemImage($item_id) {
    return ITEM_IMG_URL . (int)$item_id . '.png';
}

/**
 * Retorna URL de imagem de classe
 * Usa ícones de jobs do Divine Pride
 * Divine Pride tem IDs completos para todas as classes
 */
function classImage($class_id) {
    // Divine Pride tem os ícones com os IDs corretos do rAthena
    // Exemplo: 4079 = Shadow Chaser Trans -> icon_jobs_4079.png
    return 'https://static.divine-pride.net/images/jobs/icon_jobs_' . (int)$class_id . '.png';
}

/**
 * Retorna nome da classe baseado no class_id
 */
function className($class_id) {
    $classes = [
        0 => 'Novice',
        1 => 'Swordman', 2 => 'Mage', 3 => 'Archer', 4 => 'Acolyte', 5 => 'Merchant', 6 => 'Thief',
        7 => 'Knight', 8 => 'Priest', 9 => 'Wizard', 10 => 'Blacksmith', 11 => 'Hunter', 12 => 'Assassin',
        14 => 'Crusader', 15 => 'Monk', 16 => 'Sage', 17 => 'Rogue', 18 => 'Alchemist', 19 => 'Bard', 20 => 'Dancer',
        4008 => 'Lord Knight', 4009 => 'High Priest', 4010 => 'High Wizard', 4011 => 'Whitesmith',
        4012 => 'Sniper', 4013 => 'Assassin Cross', 4015 => 'Paladin', 4016 => 'Champion',
        4017 => 'Professor', 4018 => 'Stalker', 4019 => 'Creator', 4020 => 'Clown', 4021 => 'Gypsy',
        4054 => 'Rune Knight', 4055 => 'Warlock', 4056 => 'Ranger', 4057 => 'Arch Bishop',
        4058 => 'Mechanic', 4059 => 'Guillotine Cross',
        4060 => 'Rune Knight', 4061 => 'Warlock', 4062 => 'Ranger', 4063 => 'Arch Bishop',
        4064 => 'Mechanic', 4065 => 'Guillotine Cross',
        4066 => 'Royal Guard', 4067 => 'Sorcerer', 4068 => 'Minstrel', 4069 => 'Wanderer',
        4070 => 'Sura', 4071 => 'Genetic', 4072 => 'Shadow Chaser',
        4073 => 'Royal Guard', 4074 => 'Sorcerer', 4075 => 'Minstrel', 4076 => 'Wanderer',
        4077 => 'Sura', 4078 => 'Genetic', 4079 => 'Shadow Chaser',
        4096 => 'Super Novice',
        4190 => 'Star Emperor', 4191 => 'Soul Reaper',
        4252 => 'Dragon Knight', 4253 => 'Arch Mage', 4254 => 'Windhawk', 4255 => 'Cardinal',
        4256 => 'Meister', 4257 => 'Shadow Cross', 4258 => 'Imperial Guard', 4259 => 'Elemental Master',
        4260 => 'Troubadour', 4261 => 'Trouvere', 4262 => 'Inquisitor', 4263 => 'Biolo', 4264 => 'Abyss Chaser',
    ];
    return $classes[$class_id] ?? 'Unknown (' . $class_id . ')';
}

/**
 * Retorna URL do emblema da guild
 */
function guildEmblem($guild_id) {
    if (empty($guild_id) || $guild_id <= 0) {
        return 'assets/images/guild_placeholder.png';
    }
    return 'emblem.php?id=' . (int)$guild_id;
}

/**
 * Retorna nome da classe
 */
function getClassName($class_id) {
    global $JOB_CLASSES;
    return $JOB_CLASSES[$class_id] ?? 'Classe #' . $class_id;
}

/**
 * Retorna cor do time
 */
function getTeamColor($team) {
    global $TEAM_COLORS;
    return $TEAM_COLORS[$team]['color'] ?? '#888888';
}

/**
 * Retorna nome do time
 */
function getTeamName($team) {
    global $TEAM_COLORS;
    return $TEAM_COLORS[$team]['name'] ?? 'Time ' . $team;
}

/**
 * Formata numero grande
 */
function formatNumber($number) {
    return number_format($number ?? 0, 0, ',', '.');
}

/**
 * Formata duracao em segundos para mm:ss
 */
function formatDuration($seconds) {
    $minutes = floor($seconds / 60);
    $secs = $seconds % 60;
    return sprintf('%02d:%02d', $minutes, $secs);
}

/**
 * Formata data/hora
 */
function formatDateTime($datetime) {
    if (empty($datetime)) return '-';
    $dt = new DateTime($datetime);
    return $dt->format('d/m/Y H:i');
}

/**
 * Formata data
 */
function formatDate($datetime) {
    if (empty($datetime)) return '-';
    $dt = new DateTime($datetime);
    return $dt->format('d/m/Y');
}

/**
 * Calcula KDA
 */
function calculateKDA($kills, $deaths, $assists) {
    $deaths = max(1, $deaths);
    return round(($kills + $assists) / $deaths, 2);
}

/**
 * Calcula taxa de vitoria
 */
function calculateWinRate($wins, $total) {
    if ($total == 0) return 0;
    return round(($wins / $total) * 100, 1);
}

/**
 * Retorna classe CSS para KDA
 */
function getKdaClass($kda) {
    if ($kda >= 5) return 'kda-legendary';
    if ($kda >= 3) return 'kda-excellent';
    if ($kda >= 2) return 'kda-good';
    if ($kda >= 1) return 'kda-average';
    return 'kda-poor';
}

/**
 * Retorna classe CSS para win rate
 */
function getWinRateClass($rate) {
    if ($rate >= 70) return 'winrate-excellent';
    if ($rate >= 55) return 'winrate-good';
    if ($rate >= 45) return 'winrate-average';
    return 'winrate-poor';
}

/**
 * Gera paginacao
 */
function paginate($currentPage, $totalPages, $baseUrl) {
    if ($totalPages <= 1) return '';
    
    $html = '<nav aria-label="Navegacao"><ul class="pagination justify-content-center">';
    
    // Anterior
    if ($currentPage > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . e($baseUrl) . '&page=' . ($currentPage - 1) . '">&laquo; Anterior</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">&laquo; Anterior</span></li>';
    }
    
    // Paginas
    $start = max(1, $currentPage - 2);
    $end = min($totalPages, $currentPage + 2);
    
    if ($start > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . e($baseUrl) . '&page=1">1</a></li>';
        if ($start > 2) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
    }
    
    for ($i = $start; $i <= $end; $i++) {
        if ($i == $currentPage) {
            $html .= '<li class="page-item active"><span class="page-link">' . $i . '</span></li>';
        } else {
            $html .= '<li class="page-item"><a class="page-link" href="' . e($baseUrl) . '&page=' . $i . '">' . $i . '</a></li>';
        }
    }
    
    if ($end < $totalPages) {
        if ($end < $totalPages - 1) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
        $html .= '<li class="page-item"><a class="page-link" href="' . e($baseUrl) . '&page=' . $totalPages . '">' . $totalPages . '</a></li>';
    }
    
    // Proximo
    if ($currentPage < $totalPages) {
        $html .= '<li class="page-item"><a class="page-link" href="' . e($baseUrl) . '&page=' . ($currentPage + 1) . '">Proximo &raquo;</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">Proximo &raquo;</span></li>';
    }
    
    $html .= '</ul></nav>';
    
    return $html;
}

/**
 * Redireciona para outra pagina
 */
function redirect($url) {
    header('Location: ' . $url);
    exit;
}

/**
 * Define mensagem flash
 */
function setFlash($type, $message) {
    $_SESSION['flash'] = [
        'type' => $type,
        'message' => $message
    ];
}

/**
 * Retorna e limpa mensagem flash
 */
function getFlash() {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

/**
 * Exibe mensagem flash como HTML
 */
function displayFlash() {
    $flash = getFlash();
    if ($flash) {
        $alertClass = $flash['type'] === 'success' ? 'alert-success' : 
                     ($flash['type'] === 'error' ? 'alert-danger' : 'alert-info');
        echo '<div class="alert ' . $alertClass . ' alert-dismissible fade show" role="alert">';
        echo e($flash['message']);
        echo '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Fechar"></button>';
        echo '</div>';
    }
}

/**
 * Retorna parametro GET limpo
 */
function getParam($name, $default = null) {
    return isset($_GET[$name]) ? trim($_GET[$name]) : $default;
}

/**
 * Retorna parametro POST limpo
 */
function postParam($name, $default = null) {
    return isset($_POST[$name]) ? trim($_POST[$name]) : $default;
}

/**
 * Retorna inteiro GET
 */
function getInt($name, $default = 0) {
    return isset($_GET[$name]) ? (int)$_GET[$name] : $default;
}

/**
 * Retorna inteiro POST
 */
function postInt($name, $default = 0) {
    return isset($_POST[$name]) ? (int)$_POST[$name] : $default;
}

/**
 * Busca estatisticas gerais de um personagem
 * USA TABELA: arena7x7_ranking
 */
function getPlayerStats($char_id) {
    $sql = "SELECT 
                r.*,
                r.char_name,
                r.total_damage_done as total_damage,
                r.total_healing_done as total_healing,
                r.points as rating_points
            FROM arena7x7_ranking r
            WHERE r.char_id = :char_id";
    
    $result = db()->fetchOne($sql, ['char_id' => $char_id]);
    
    if ($result) {
        // Buscar classe do personagem na tabela char
        $charInfo = db()->fetchOne(
            "SELECT class, base_level FROM `" . TABLE_CHAR . "` WHERE char_id = :char_id",
            ['char_id' => $char_id]
        );
        if ($charInfo) {
            $result['class'] = $charInfo['class'];
            $result['base_level'] = $charInfo['base_level'];
        }
    }
    
    return $result;
}

/**
 * Busca top jogadores por pontuacao
 * USA TABELA: arena7x7_ranking
 */
function getTopPlayers($limit = 10, $offset = 0) {
    $sql = "SELECT 
                r.*,
                r.char_name,
                r.points as rating_points,
                r.total_damage_done as total_damage,
                r.total_healing_done as total_healing,
                r.ties,
                c.class,
                c.base_level
            FROM arena7x7_ranking r
            LEFT JOIN `" . TABLE_CHAR . "` c ON c.char_id = r.char_id
            WHERE r.matches_played > 0
            ORDER BY r.points DESC, r.wins DESC
            LIMIT " . (int)$limit . " OFFSET " . (int)$offset;
    
    return db()->fetchAll($sql);
}

/**
 * Conta total de jogadores com partidas
 */
function countPlayersWithMatches() {
    $sql = "SELECT COUNT(*) FROM arena7x7_ranking WHERE matches_played > 0";
    return db()->fetchColumn($sql);
}

/**
 * Busca partidas recentes
 * USA TABELA: arena7x7_matches (colunas blue_score, red_score, blue_guild_name, red_guild_name)
 */
function getRecentMatches($limit = 10, $offset = 0, $filters = []) {
    $where = ['1=1'];
    $params = [];
    
    if (!empty($filters['date_from'])) {
        $where[] = "m.start_time >= :date_from";
        $params['date_from'] = $filters['date_from'] . ' 00:00:00';
    }
    
    if (!empty($filters['date_to'])) {
        $where[] = "m.start_time <= :date_to";
        $params['date_to'] = $filters['date_to'] . ' 23:59:59';
    }
    
    if (!empty($filters['char_id'])) {
        $where[] = "m.match_id IN (SELECT match_id FROM arena7x7_match_players WHERE char_id = :char_id)";
        $params['char_id'] = $filters['char_id'];
    }
    
    $whereStr = implode(' AND ', $where);
    
    $sql = "SELECT m.*,
                   m.blue_score as team_blue_score,
                   m.red_score as team_red_score,
                   m.blue_guild_id,
                   m.red_guild_id,
                   COALESCE(NULLIF(m.blue_guild_name, ''), 'Time Azul') as blue_guild_name,
                   COALESCE(NULLIF(m.red_guild_name, ''), 'Time Vermelho') as red_guild_name
            FROM arena7x7_matches m
            WHERE $whereStr
            ORDER BY m.start_time DESC
            LIMIT " . (int)$limit . " OFFSET " . (int)$offset;
    
    return db()->fetchAll($sql, $params);
}

/**
 * Conta partidas com filtros
 */
function countMatches($filters = []) {
    $where = ['1=1'];
    $params = [];
    
    if (!empty($filters['date_from'])) {
        $where[] = "start_time >= :date_from";
        $params['date_from'] = $filters['date_from'] . ' 00:00:00';
    }
    
    if (!empty($filters['date_to'])) {
        $where[] = "start_time <= :date_to";
        $params['date_to'] = $filters['date_to'] . ' 23:59:59';
    }
    
    if (!empty($filters['char_id'])) {
        $where[] = "match_id IN (SELECT match_id FROM arena7x7_match_players WHERE char_id = :char_id)";
        $params['char_id'] = $filters['char_id'];
    }
    
    $whereStr = implode(' AND ', $where);
    
    $sql = "SELECT COUNT(*) FROM arena7x7_matches WHERE $whereStr";
    return db()->fetchColumn($sql, $params);
}

/**
 * Busca detalhes de uma partida
 * Inclui nomes das guilds dos times
 */
function getMatchDetails($match_id) {
    $sql = "SELECT m.*,
                   m.blue_score as team_blue_score,
                   m.red_score as team_red_score,
                   COALESCE(NULLIF(m.blue_guild_name, ''), 'Time Azul') as blue_guild_name,
                   COALESCE(NULLIF(m.red_guild_name, ''), 'Time Vermelho') as red_guild_name
            FROM arena7x7_matches m 
            WHERE m.match_id = :match_id";
    return db()->fetchOne($sql, ['match_id' => $match_id]);
}

/**
 * Busca jogadores de uma partida com estatisticas
 * USA TABELAS: arena7x7_match_players + arena7x7_match_stats
 */
function getMatchPlayers($match_id) {
    $sql = "SELECT 
                mp.char_id,
                mp.char_name,
                mp.team,
                mp.job_class as class,
                mp.base_level,
                mp.is_winner,
                COALESCE(mp.guild_id, 0) as guild_id,
                COALESCE(mp.guild_name, '') as guild_name,
                COALESCE(ms.kills, 0) as kills,
                COALESCE(ms.deaths, 0) as deaths,
                COALESCE(ms.assists, 0) as assists,
                COALESCE(ms.damage_done, 0) as total_damage,
                COALESCE(ms.healing_done, 0) as total_healing,
                COALESCE(ms.damage_received, 0) as damage_taken
            FROM arena7x7_match_players mp
            LEFT JOIN arena7x7_match_stats ms ON ms.match_id = mp.match_id AND ms.char_id = mp.char_id
            WHERE mp.match_id = :match_id
            ORDER BY mp.team, ms.damage_done DESC";
    
    return db()->fetchAll($sql, ['match_id' => $match_id]);
}

/**
 * Busca kills de uma partida
 * USA TABELA: arena7x7_kill_log (nova tabela com nomes)
 */
function getMatchKills($match_id) {
    // Primeiro tentar a nova tabela com nomes
    $sql = "SELECT 
                kl.killer_id,
                kl.killer_name,
                kl.victim_id,
                kl.victim_name,
                kl.skill_id as kill_skill_id,
                kl.skill_name,
                kl.final_damage,
                kl.killer_class,
                kl.victim_class,
                kl.assist1_id,
                kl.assist1_name,
                kl.assist2_id,
                kl.assist2_name,
                kl.timestamp
            FROM arena7x7_kill_log kl
            WHERE kl.match_id = :match_id
            ORDER BY kl.timestamp ASC";
    
    $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    
    // Se não houver dados na nova tabela, tentar a tabela antiga
    if (empty($result)) {
        $sql = "SELECT 
                    k.*,
                    killer.name as killer_name,
                    victim.name as victim_name,
                    0 as final_damage
                FROM arena7x7_kills k
                JOIN `" . TABLE_CHAR . "` killer ON killer.char_id = k.killer_id
                JOIN `" . TABLE_CHAR . "` victim ON victim.char_id = k.victim_id
                WHERE k.match_id = :match_id
                ORDER BY k.id ASC";
        
        $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    }
    
    return $result;
}

/**
 * Busca skills usadas em uma partida com detalhes do alvo
 * USA TABELA: arena7x7_skill_summary (nova) ou arena7x7_damage_by_skill (antiga)
 */
function getMatchSkills($match_id) {
    // Primeiro tentar a nova tabela skill_summary
    $sql = "SELECT 
                ss.skill_id,
                ss.skill_name,
                ss.char_id,
                ss.char_name,
                ss.total_damage as damage_dealt,
                ss.use_count as times_used,
                ss.total_hits as hit_count,
                ss.kills_with_skill as kill_count
            FROM arena7x7_skill_summary ss
            WHERE ss.match_id = :match_id
            ORDER BY ss.total_damage DESC
            LIMIT 50";
    
    $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    
    // Se não houver dados, tentar a tabela antiga
    if (empty($result)) {
        $sql = "SELECT 
                    ds.skill_id,
                    ds.skill_name,
                    ds.char_id,
                    c.name as char_name,
                    ds.total_damage as damage_dealt,
                    ds.hit_count as times_used,
                    ds.kill_count
                FROM arena7x7_damage_by_skill ds
                LEFT JOIN `" . TABLE_CHAR . "` c ON c.char_id = ds.char_id
                WHERE ds.match_id = :match_id
                ORDER BY ds.total_damage DESC
                LIMIT 50";
        
        $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    }
    
    return $result;
}

/**
 * Busca log detalhado de skills com dano por target
 * USA TABELA: arena7x7_skill_log (nova)
 */
function getMatchSkillLog($match_id) {
    $sql = "SELECT 
                sl.caster_id,
                sl.caster_name,
                sl.target_id,
                sl.target_name,
                sl.skill_id,
                sl.skill_name,
                sl.skill_level,
                sl.damage,
                sl.hits,
                sl.is_critical,
                sl.is_kill,
                sl.timestamp
            FROM arena7x7_skill_log sl
            WHERE sl.match_id = :match_id
            ORDER BY sl.timestamp ASC";
    
    return db()->fetchAll($sql, ['match_id' => $match_id]);
}

/**
 * Busca dano agregado por alvo
 * USA TABELA: arena7x7_damage_by_target (nova)
 */
function getMatchDamageByTarget($match_id) {
    $sql = "SELECT 
                dt.attacker_id,
                COALESCE(NULLIF(dt.attacker_name, ''), ca.name) as attacker_name,
                dt.target_id,
                COALESCE(NULLIF(dt.target_name, ''), ct.name) as target_name,
                dt.skill_damage,
                dt.normal_damage,
                dt.total_damage
            FROM arena7x7_damage_by_target dt
            LEFT JOIN `" . TABLE_CHAR . "` ca ON ca.char_id = dt.attacker_id
            LEFT JOIN `" . TABLE_CHAR . "` ct ON ct.char_id = dt.target_id
            WHERE dt.match_id = :match_id
            ORDER BY dt.total_damage DESC";
    
    return db()->fetchAll($sql, ['match_id' => $match_id]);
}

/**
 * Busca ataques normais de uma partida
 * USA TABELA: arena7x7_attack_log (nova)
 */
function getMatchAttacks($match_id) {
    $sql = "SELECT 
                al.attacker_id,
                al.attacker_name,
                al.target_id,
                al.target_name,
                al.total_damage,
                al.hit_count,
                al.critical_count,
                al.miss_count
            FROM arena7x7_attack_log al
            WHERE al.match_id = :match_id
            ORDER BY al.total_damage DESC";
    
    return db()->fetchAll($sql, ['match_id' => $match_id]);
}

/**
 * Busca itens consumidos em uma partida
 * USA TABELA: arena7x7_item_summary ou arena7x7_item_log (nova)
 */
function getMatchItems($match_id) {
    // Primeiro tentar item_summary
    $sql = "SELECT 
                isu.char_id,
                isu.char_name,
                isu.item_id,
                isu.item_name,
                isu.item_type,
                isu.total_amount as amount
            FROM arena7x7_item_summary isu
            WHERE isu.match_id = :match_id
            ORDER BY isu.char_name, isu.item_type, isu.total_amount DESC";
    
    $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    
    // Se não houver dados, tentar item_log
    if (empty($result)) {
        $sql = "SELECT 
                    il.char_id,
                    il.char_name,
                    il.item_id,
                    il.item_name,
                    il.item_type,
                    SUM(il.amount) as amount
                FROM arena7x7_item_log il
                WHERE il.match_id = :match_id
                GROUP BY il.char_id, il.char_name, il.item_id, il.item_name, il.item_type
                ORDER BY il.char_name, il.item_type, amount DESC";
        
        $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    }
    
    return $result;
}

/**
 * Busca curas de uma partida
 * USA TABELA: arena7x7_heal_log (nova) ou arena7x7_support_by_target (antiga)
 */
function getMatchHeals($match_id) {
    // Primeiro tentar a nova tabela heal_log
    $sql = "SELECT 
                hl.healer_id,
                hl.healer_name,
                hl.target_id,
                hl.target_name,
                hl.skill_id,
                hl.skill_name,
                hl.heal_amount
            FROM arena7x7_heal_log hl
            WHERE hl.match_id = :match_id
            ORDER BY hl.heal_amount DESC
            LIMIT 50";
    
    $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    
    // Se não houver dados, tentar a tabela antiga
    if (empty($result)) {
        $sql = "SELECT 
                    st.caster_id as healer_id,
                    st.target_id,
                    st.total_healing as heal_amount,
                    healer.name as healer_name,
                    target.name as target_name,
                    0 as skill_id,
                    '' as skill_name
                FROM arena7x7_support_by_target st
                JOIN `" . TABLE_CHAR . "` healer ON healer.char_id = st.caster_id
                JOIN `" . TABLE_CHAR . "` target ON target.char_id = st.target_id
                WHERE st.match_id = :match_id AND st.total_healing > 0
                ORDER BY st.total_healing DESC
                LIMIT 50";
        
        $result = db()->fetchAll($sql, ['match_id' => $match_id]);
    }
    
    return $result;
}

/**
 * Busca assistencias de uma partida
 * USA dados da tabela arena7x7_kills (assist1_id, assist2_id, assist3_id)
 */
function getMatchAssists($match_id) {
    // Buscar kills que tem assists
    $sql = "SELECT 
                k.killer_id,
                k.victim_id,
                k.assist1_id,
                k.assist2_id,
                k.assist3_id,
                k.timestamp as assist_time,
                killer.name as killer_name,
                victim.name as victim_name
            FROM arena7x7_kills k
            JOIN `" . TABLE_CHAR . "` killer ON killer.char_id = k.killer_id
            JOIN `" . TABLE_CHAR . "` victim ON victim.char_id = k.victim_id
            WHERE k.match_id = :match_id 
              AND (k.assist1_id IS NOT NULL OR k.assist2_id IS NOT NULL OR k.assist3_id IS NOT NULL)
            ORDER BY k.timestamp ASC";
    
    $kills = db()->fetchAll($sql, ['match_id' => $match_id]);
    
    // Transformar em lista de assists
    $assists = [];
    foreach ($kills as $kill) {
        foreach (['assist1_id', 'assist2_id', 'assist3_id'] as $assistField) {
            if (!empty($kill[$assistField])) {
                $assistName = db()->fetchColumn(
                    "SELECT name FROM `" . TABLE_CHAR . "` WHERE char_id = :id",
                    ['id' => $kill[$assistField]]
                );
                $assists[] = [
                    'assistant_id' => $kill[$assistField],
                    'assistant_name' => $assistName ?? 'Desconhecido',
                    'killer_id' => $kill['killer_id'],
                    'killer_name' => $kill['killer_name'],
                    'victim_id' => $kill['victim_id'],
                    'victim_name' => $kill['victim_name'],
                    'assist_time' => $kill['assist_time']
                ];
            }
        }
    }
    
    return $assists;
}

/**
 * Busca historico de partidas de um jogador
 */
function getPlayerMatchHistory($char_id, $limit = 20, $offset = 0) {
    $sql = "SELECT 
                m.match_id,
                m.start_time,
                m.end_time,
                m.duration,
                m.winner_team,
                m.blue_score as team_blue_score,
                m.red_score as team_red_score,
                m.map_name,
                mp.team,
                mp.is_winner,
                COALESCE(ms.kills, 0) as kills,
                COALESCE(ms.deaths, 0) as deaths,
                COALESCE(ms.assists, 0) as assists,
                COALESCE(ms.damage_done, 0) as total_damage,
                COALESCE(ms.healing_done, 0) as total_healing
            FROM arena7x7_match_players mp
            JOIN arena7x7_matches m ON m.match_id = mp.match_id
            LEFT JOIN arena7x7_match_stats ms ON ms.match_id = mp.match_id AND ms.char_id = mp.char_id
            WHERE mp.char_id = :char_id
            ORDER BY m.start_time DESC
            LIMIT " . (int)$limit . " OFFSET " . (int)$offset;
    
    return db()->fetchAll($sql, ['char_id' => $char_id]);
}

/**
 * Conta partidas de um jogador
 */
function countPlayerMatches($char_id) {
    $sql = "SELECT COUNT(*) FROM arena7x7_match_players WHERE char_id = :char_id";
    return db()->fetchColumn($sql, ['char_id' => $char_id]);
}

/**
 * Busca jogadores por nome
 */
function searchPlayers($name, $limit = 20) {
    $sql = "SELECT 
                r.*,
                r.points as rating_points,
                r.total_damage_done as total_damage,
                r.total_healing_done as total_healing,
                c.class,
                c.base_level
            FROM arena7x7_ranking r
            LEFT JOIN `" . TABLE_CHAR . "` c ON c.char_id = r.char_id
            WHERE r.char_name LIKE :name
            ORDER BY r.points DESC
            LIMIT " . (int)$limit;
    
    return db()->fetchAll($sql, ['name' => '%' . $name . '%']);
}

/**
 * Busca dados do personagem logado
 * USA: arena7x7_ranking
 */
function getMyCharacters($account_id) {
    $sql = "SELECT 
                c.char_id,
                c.name as char_name,
                c.class,
                c.base_level,
                COALESCE(r.points, 1000) as rating_points,
                COALESCE(r.matches_played, 0) as matches_played,
                COALESCE(r.wins, 0) as wins,
                COALESCE(r.losses, 0) as losses,
                COALESCE(r.total_kills, 0) as total_kills,
                COALESCE(r.total_deaths, 0) as total_deaths,
                COALESCE(r.total_assists, 0) as total_assists,
                COALESCE(r.total_damage_done, 0) as total_damage,
                COALESCE(r.total_healing_done, 0) as total_healing
            FROM `" . TABLE_CHAR . "` c
            LEFT JOIN arena7x7_ranking r ON r.char_id = c.char_id
            WHERE c.account_id = :account_id
            ORDER BY r.points DESC, c.base_level DESC";
    
    return db()->fetchAll($sql, ['account_id' => $account_id]);
}

/**
 * Ranking por classe
 * USA TABELA: arena7x7_ranking_by_class
 */
function getRankingByClass($class_id = null, $limit = 10, $offset = 0) {
    if ($class_id) {
        $sql = "SELECT 
                    rbc.*,
                    c.name as char_name,
                    c.base_level
                FROM arena7x7_ranking_by_class rbc
                JOIN `" . TABLE_CHAR . "` c ON c.char_id = rbc.char_id
                WHERE rbc.job_class = :class_id
                ORDER BY rbc.points DESC
                LIMIT " . (int)$limit . " OFFSET " . (int)$offset;
        return db()->fetchAll($sql, ['class_id' => $class_id]);
    }
    
    // Top de cada classe
    $sql = "SELECT 
                rbc.*,
                c.name as char_name,
                c.base_level
            FROM arena7x7_ranking_by_class rbc
            JOIN `" . TABLE_CHAR . "` c ON c.char_id = rbc.char_id
            INNER JOIN (
                SELECT job_class, MAX(points) as max_points
                FROM arena7x7_ranking_by_class
                GROUP BY job_class
            ) top ON rbc.job_class = top.job_class AND rbc.points = top.max_points
            ORDER BY rbc.points DESC
            LIMIT " . (int)$limit;
    
    return db()->fetchAll($sql);
}

/**
 * Busca estatisticas globais
 */
function getGlobalStats() {
    $stats = [];
    
    // Total de partidas
    $stats['total_matches'] = db()->fetchColumn("SELECT COUNT(*) FROM arena7x7_matches") ?? 0;
    
    // Total de jogadores unicos
    $stats['total_players'] = db()->fetchColumn("SELECT COUNT(*) FROM arena7x7_ranking WHERE matches_played > 0") ?? 0;
    
    // Total de kills
    $stats['total_kills'] = db()->fetchColumn("SELECT COUNT(*) FROM arena7x7_kills") ?? 0;
    
    // Maior dano em uma partida
    $stats['max_damage'] = db()->fetchColumn("SELECT MAX(damage_done) FROM arena7x7_match_stats") ?? 0;
    
    // Maior heal em uma partida
    $stats['max_healing'] = db()->fetchColumn("SELECT MAX(healing_done) FROM arena7x7_match_stats") ?? 0;
    
    // Top killer
    $topKiller = db()->fetchOne("SELECT char_name, total_kills FROM arena7x7_ranking ORDER BY total_kills DESC LIMIT 1");
    $stats['top_killer'] = $topKiller['char_name'] ?? '-';
    $stats['top_killer_count'] = $topKiller['total_kills'] ?? 0;
    
    // Top healer
    $topHealer = db()->fetchOne("SELECT char_name, total_healing_done FROM arena7x7_ranking ORDER BY total_healing_done DESC LIMIT 1");
    $stats['top_healer'] = $topHealer['char_name'] ?? '-';
    $stats['top_healer_amount'] = $topHealer['total_healing_done'] ?? 0;
    
    return $stats;
}

/**
 * Busca temporadas
 * USA TABELA: arena7x7_seasons
 */
function getSeasons() {
    $sql = "SELECT * FROM arena7x7_seasons ORDER BY season_id DESC";
    return db()->fetchAll($sql);
}

/**
 * Busca temporada atual
 */
function getCurrentSeason() {
    $sql = "SELECT * FROM arena7x7_seasons WHERE status = 'active' ORDER BY season_id DESC LIMIT 1";
    return db()->fetchOne($sql);
}

/**
 * Busca ranking de uma temporada especifica
 */
function getSeasonRanking($season_id = null, $limit = 10, $offset = 0) {
    if ($season_id === null) {
        $currentSeason = getCurrentSeason();
        $season_id = $currentSeason ? $currentSeason['season_id'] : 1;
    }
    
    $sql = "SELECT 
                r.*,
                r.points as rating_points,
                r.total_damage_done as total_damage,
                r.total_healing_done as total_healing,
                c.class,
                c.base_level
            FROM arena7x7_ranking r
            LEFT JOIN `" . TABLE_CHAR . "` c ON c.char_id = r.char_id
            WHERE r.matches_played > 0
            ORDER BY r.points DESC
            LIMIT " . (int)$limit . " OFFSET " . (int)$offset;
    
    return db()->fetchAll($sql);
}

/**
 * Busca jogadores para o perfil com detalhes extras
 */
function getPlayerProfile($char_id) {
    $player = getPlayerStats($char_id);
    if (!$player) return null;
    
    // Adicionar estatisticas extras
    $player['win_rate'] = calculateWinRate($player['wins'] ?? 0, $player['matches_played'] ?? 0);
    $player['kda'] = calculateKDA(
        $player['total_kills'] ?? 0, 
        $player['total_deaths'] ?? 0, 
        $player['total_assists'] ?? 0
    );
    
    // Buscar posicao no ranking
    $sql = "SELECT COUNT(*) + 1 as position 
            FROM arena7x7_ranking 
            WHERE points > (SELECT points FROM arena7x7_ranking WHERE char_id = :char_id)";
    $rankPos = db()->fetchOne($sql, ['char_id' => $char_id]);
    $player['rank_position'] = $rankPos['position'] ?? 1;
    
    // Ultimas 5 partidas
    $player['recent_matches'] = getPlayerMatchHistory($char_id, 5);
    
    // Skills mais usadas (se houver dados)
    $sql = "SELECT 
                ds.skill_id,
                ds.skill_name,
                SUM(ds.total_damage) as total_damage,
                SUM(ds.hit_count) as times_used,
                SUM(ds.kill_count) as kills_with_skill
            FROM arena7x7_damage_by_skill ds
            JOIN arena7x7_match_players mp ON mp.match_id = ds.match_id AND mp.char_id = ds.char_id
            WHERE ds.char_id = :char_id
            GROUP BY ds.skill_id, ds.skill_name
            ORDER BY total_damage DESC
            LIMIT 10";
    $player['top_skills'] = db()->fetchAll($sql, ['char_id' => $char_id]);
    
    return $player;
}

/**
 * Verifica se usuario tem permissao de admin
 */
function isAdmin() {
    // Por enquanto, sem sistema de admin
    return false;
}

/**
 * Gera token CSRF
 */
function generateCsrfToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Valida token CSRF
 */
function validateCsrfToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Retorna campo hidden com token CSRF
 */
function csrfField() {
    return '<input type="hidden" name="csrf_token" value="' . e(generateCsrfToken()) . '">';
}
