<?php
/**
 * Arena 7x7 - Lista de Partidas
 */

define('ARENA7X7', true);
require_once 'includes/config.php';
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

Auth::startSession();

$pageTitle = 'Partidas';

// Filtros
$filters = [];
$dateFrom = getParam('date_from');
$dateTo = getParam('date_to');

if ($dateFrom) $filters['date_from'] = $dateFrom;
if ($dateTo) $filters['date_to'] = $dateTo;

// Paginacao
$page = max(1, getInt('page', 1));
$perPage = ITEMS_PER_PAGE;
$offset = ($page - 1) * $perPage;

// Total de partidas
$totalMatches = countMatches($filters);
$totalPages = ceil($totalMatches / $perPage);

// Buscar partidas
$matches = getRecentMatches($perPage, $offset, $filters);

include 'includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="mb-1"><i class="bi bi-controller me-2 text-primary"></i>Partidas</h1>
        <p class="text-muted mb-0">Historico de todas as partidas da Arena 7x7</p>
    </div>
    <div>
        <span class="badge bg-primary fs-6">
            <i class="bi bi-joystick me-1"></i><?= formatNumber($totalMatches) ?> partidas
        </span>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" action="" class="row g-3 align-items-end">
            <div class="col-md-3">
                <label for="date_from" class="form-label">Data Inicial</label>
                <input type="date" 
                       class="form-control" 
                       id="date_from" 
                       name="date_from"
                       value="<?= e($dateFrom) ?>">
            </div>
            <div class="col-md-3">
                <label for="date_to" class="form-label">Data Final</label>
                <input type="date" 
                       class="form-control" 
                       id="date_to" 
                       name="date_to"
                       value="<?= e($dateTo) ?>">
            </div>
            <div class="col-md-3">
                <button type="submit" class="btn btn-primary w-100">
                    <i class="bi bi-search me-2"></i>Filtrar
                </button>
            </div>
            <div class="col-md-3">
                <a href="matches.php" class="btn btn-outline-secondary w-100">
                    <i class="bi bi-x-circle me-2"></i>Limpar Filtros
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Lista de Partidas -->
<?php if (empty($matches)): ?>
    <div class="card">
        <div class="card-body text-center py-5">
            <i class="bi bi-emoji-frown fs-1 d-block mb-3 text-muted"></i>
            <h5 class="text-muted">Nenhuma partida encontrada</h5>
            <p class="text-muted mb-0">
                <?php if (!empty($filters)): ?>
                    Tente ajustar os filtros de busca.
                <?php else: ?>
                    As partidas aparecerao aqui quando forem jogadas.
                <?php endif; ?>
            </p>
        </div>
    </div>
<?php else: ?>
    <div class="row g-4">
        <?php foreach ($matches as $match): ?>
            <?php 
                // Buscar jogadores da partida para preview
                $matchPlayers = getMatchPlayers($match['match_id']);
                $teamBlue = array_filter($matchPlayers, fn($p) => $p['team'] == 1);
                $teamRed = array_filter($matchPlayers, fn($p) => $p['team'] == 2);
            ?>
            <div class="col-lg-6">
                <a href="<?= matchUrl($match['match_id']) ?>" class="text-decoration-none">
                    <div class="card match-card h-100">
                        <div class="match-header">
                            <span class="match-id">
                                <i class="bi bi-controller me-1"></i>Partida
                            </span>
                            <span class="match-date">
                                <i class="bi bi-calendar me-1"></i>
                                <?= formatDateTime($match['start_time']) ?>
                            </span>
                        </div>
                        
                        <div class="match-score mb-3">
                            <div class="team-score team-blue">
                                <small class="d-block text-secondary mb-1">
                                    <img src="<?= guildEmblem($match['blue_guild_id']) ?>" 
                                         alt="" class="guild-emblem-sm me-1"
                                         onerror="this.style.display='none'">
                                    <?= e($match['blue_guild_name']) ?>
                                </small>
                                <?= $match['team_blue_score'] ?>
                            </div>
                            <div class="vs">VS</div>
                            <div class="team-score team-red">
                                <small class="d-block text-secondary mb-1">
                                    <?= e($match['red_guild_name']) ?>
                                    <img src="<?= guildEmblem($match['red_guild_id']) ?>" 
                                         alt="" class="guild-emblem-sm ms-1"
                                         onerror="this.style.display='none'">
                                </small>
                                <?= $match['team_red_score'] ?>
                            </div>
                        </div>
                        
                        <div class="row g-2">
                            <!-- <?= e($match['blue_guild_name']) ?> -->
                            <div class="col-6">
                                <div class="team-blue p-2 rounded">
                                    <div class="d-flex flex-column gap-1">
                                        <?php foreach ($teamBlue as $player): ?>
                                            <div class="d-flex align-items-center gap-2 player-row">
                                                <img src="<?= classImage($player['class']) ?>" 
                                                     alt="<?= e($player['char_name']) ?>"
                                                     class="player-class-icon"
                                                     onerror="this.src='assets/images/placeholder.png'">
                                                <div class="player-info flex-grow-1 text-start">
                                                    <div class="player-name text-truncate"><?= e($player['char_name']) ?></div>
                                                    <small class="player-class text-muted"><?= className($player['class']) ?></small>
                                                </div>
                                                <div class="player-kda text-end">
                                                    <small class="text-success"><?= $player['kills'] ?></small>/<small class="text-danger"><?= $player['deaths'] ?></small>/<small class="text-info"><?= $player['assists'] ?></small>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            <!-- <?= e($match['red_guild_name']) ?> -->
                            <div class="col-6">
                                <div class="team-red p-2 rounded">
                                    <div class="d-flex flex-column gap-1">
                                        <?php foreach ($teamRed as $player): ?>
                                            <div class="d-flex align-items-center gap-2 player-row">
                                                <img src="<?= classImage($player['class']) ?>" 
                                                     alt="<?= e($player['char_name']) ?>"
                                                     class="player-class-icon"
                                                     onerror="this.src='assets/images/placeholder.png'">
                                                <div class="player-info flex-grow-1 text-start">
                                                    <div class="player-name text-truncate"><?= e($player['char_name']) ?></div>
                                                    <small class="player-class text-muted"><?= className($player['class']) ?></small>
                                                </div>
                                                <div class="player-kda text-end">
                                                    <small class="text-success"><?= $player['kills'] ?></small>/<small class="text-danger"><?= $player['deaths'] ?></small>/<small class="text-info"><?= $player['assists'] ?></small>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="text-center mt-3">
                            <?php if ($match['winner_team'] == 1): ?>
                                <span class="match-result match-result-victory">
                                    <i class="bi bi-trophy me-1"></i><?= e($match['blue_guild_name']) ?> Venceu
                                </span>
                            <?php elseif ($match['winner_team'] == 2): ?>
                                <span class="match-result match-result-defeat">
                                    <i class="bi bi-trophy me-1"></i><?= e($match['red_guild_name']) ?> Venceu
                                </span>
                            <?php else: ?>
                                <span class="match-result match-result-draw">
                                    <i class="bi bi-dash-circle me-1"></i>Empate
                                </span>
                            <?php endif; ?>
                            
                            <?php if (!empty($match['duration'])): ?>
                                <span class="badge bg-secondary ms-2">
                                    <i class="bi bi-clock me-1"></i>
                                    <?= formatDuration($match['duration']) ?>
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </a>
            </div>
        <?php endforeach; ?>
    </div>
    
    <!-- Paginacao -->
    <?php if ($totalPages > 1): ?>
        <?php 
            $queryParams = http_build_query(array_filter([
                'date_from' => $dateFrom,
                'date_to' => $dateTo
            ]));
        ?>
        <div class="mt-4">
            <?= paginate($page, $totalPages, 'matches.php?' . $queryParams) ?>
        </div>
    <?php endif; ?>
<?php endif; ?>

<?php include 'includes/footer.php'; ?>
