<?php
/**
 * Arena 7x7 - Minhas Partidas (Requer Login)
 */

define('ARENA7X7', true);
require_once 'includes/config.php';
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

Auth::startSession();

// Verificar login
if (!Auth::isLoggedIn()) {
    setFlash('error', 'Voce precisa estar logado para ver suas partidas.');
    redirect('login.php');
}

$pageTitle = 'Minhas Partidas';
$user = Auth::getUser();

// Buscar personagens do usuario
$characters = Auth::getCharacters();
$charIds = array_column($characters, 'char_id');

// Filtros
$selectedChar = getInt('char_id');
$dateFrom = getParam('date_from');
$dateTo = getParam('date_to');

$filters = [];
if ($selectedChar && in_array($selectedChar, $charIds)) {
    $filters['char_id'] = $selectedChar;
} elseif (!empty($charIds)) {
    // Se nenhum char selecionado, buscar todos do usuario
    $filters['char_ids'] = $charIds;
}

if ($dateFrom) $filters['date_from'] = $dateFrom;
if ($dateTo) $filters['date_to'] = $dateTo;

// Funcao customizada para buscar partidas dos chars do usuario
function getMyMatches($charIds, $limit = 20, $offset = 0, $filters = []) {
    if (empty($charIds)) return [];
    
    $where = ["mp.char_id IN (" . implode(',', array_map('intval', $charIds)) . ")"];
    $params = [];
    
    if (!empty($filters['char_id'])) {
        $where = ["mp.char_id = :char_id"];
        $params['char_id'] = $filters['char_id'];
    }
    
    if (!empty($filters['date_from'])) {
        $where[] = "m.start_time >= :date_from";
        $params['date_from'] = $filters['date_from'] . ' 00:00:00';
    }
    
    if (!empty($filters['date_to'])) {
        $where[] = "m.start_time <= :date_to";
        $params['date_to'] = $filters['date_to'] . ' 23:59:59';
    }
    
    $whereStr = implode(' AND ', $where);
    
    $sql = "SELECT DISTINCT m.*, mp.char_id as my_char_id, mp.team as my_team, mp.kills, mp.deaths, mp.assists,
                   c.name as my_char_name, c.class as my_class
            FROM arena7x7_matches m
            JOIN arena7x7_match_players mp ON mp.match_id = m.match_id
            JOIN `" . TABLE_CHAR . "` c ON c.char_id = mp.char_id
            WHERE $whereStr
            ORDER BY m.start_time DESC
            LIMIT :limit OFFSET :offset";
    
    $params['limit'] = $limit;
    $params['offset'] = $offset;
    
    return db()->fetchAll($sql, $params);
}

function countMyMatches($charIds, $filters = []) {
    if (empty($charIds)) return 0;
    
    $where = ["mp.char_id IN (" . implode(',', array_map('intval', $charIds)) . ")"];
    $params = [];
    
    if (!empty($filters['char_id'])) {
        $where = ["mp.char_id = :char_id"];
        $params['char_id'] = $filters['char_id'];
    }
    
    if (!empty($filters['date_from'])) {
        $where[] = "m.start_time >= :date_from";
        $params['date_from'] = $filters['date_from'] . ' 00:00:00';
    }
    
    if (!empty($filters['date_to'])) {
        $where[] = "m.start_time <= :date_to";
        $params['date_to'] = $filters['date_to'] . ' 23:59:59';
    }
    
    $whereStr = implode(' AND ', $where);
    
    $sql = "SELECT COUNT(DISTINCT m.match_id)
            FROM arena7x7_matches m
            JOIN arena7x7_match_players mp ON mp.match_id = m.match_id
            WHERE $whereStr";
    
    return db()->fetchColumn($sql, $params);
}

// Paginacao
$page = max(1, getInt('page', 1));
$perPage = ITEMS_PER_PAGE;
$offset = ($page - 1) * $perPage;

// Total de partidas
$totalMatches = countMyMatches($charIds, $filters);
$totalPages = ceil($totalMatches / $perPage);

// Buscar partidas
$matches = getMyMatches($charIds, $perPage, $offset, $filters);

// Estatisticas gerais
$totalWins = 0;
$totalLosses = 0;
$totalKills = 0;
$totalDeaths = 0;
$totalAssists = 0;

foreach ($characters as $char) {
    $stats = getPlayerStats($char['char_id']);
    if ($stats) {
        $totalWins += $stats['wins'];
        $totalLosses += $stats['losses'];
        $totalKills += $stats['total_kills'];
        $totalDeaths += $stats['total_deaths'];
        $totalAssists += $stats['total_assists'];
    }
}

$overallWinRate = calculateWinRate($totalWins, $totalWins + $totalLosses);
$overallKda = calculateKDA($totalKills, $totalDeaths, $totalAssists);

include 'includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="mb-1"><i class="bi bi-person-badge me-2 text-primary"></i>Minhas Partidas</h1>
        <p class="text-muted mb-0">Historico de partidas da conta <strong><?= e($user['userid']) ?></strong></p>
    </div>
</div>

<!-- Resumo Geral -->
<div class="row g-4 mb-4">
    <div class="col-md-2 col-6">
        <div class="card stat-card">
            <i class="bi bi-people stat-icon text-info"></i>
            <div class="stat-value"><?= count($characters) ?></div>
            <div class="stat-label">Personagens</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card stat-rating">
            <i class="bi bi-controller stat-icon"></i>
            <div class="stat-value"><?= formatNumber($totalMatches) ?></div>
            <div class="stat-label">Partidas</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card stat-wins">
            <i class="bi bi-trophy stat-icon"></i>
            <div class="stat-value"><?= $totalWins ?></div>
            <div class="stat-label">Vitorias</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card">
            <i class="bi bi-percent stat-icon text-warning"></i>
            <div class="stat-value <?= getWinRateClass($overallWinRate) ?>"><?= $overallWinRate ?>%</div>
            <div class="stat-label">Win Rate</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card stat-kills">
            <i class="bi bi-lightning stat-icon"></i>
            <div class="stat-value"><?= $totalKills ?></div>
            <div class="stat-label">Kills</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card">
            <i class="bi bi-graph-up stat-icon text-success"></i>
            <div class="stat-value <?= getKdaClass($overallKda) ?>"><?= number_format($overallKda, 2) ?></div>
            <div class="stat-label">KDA</div>
        </div>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" action="" class="row g-3 align-items-end">
            <div class="col-md-3">
                <label for="char_id" class="form-label">Personagem</label>
                <select class="form-select" id="char_id" name="char_id">
                    <option value="">Todos os personagens</option>
                    <?php foreach ($characters as $char): ?>
                        <option value="<?= $char['char_id'] ?>" <?= $selectedChar == $char['char_id'] ? 'selected' : '' ?>>
                            <?= e($char['name']) ?> (<?= e(getClassName($char['class'])) ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <label for="date_from" class="form-label">Data Inicial</label>
                <input type="date" class="form-control" id="date_from" name="date_from" value="<?= e($dateFrom) ?>">
            </div>
            <div class="col-md-2">
                <label for="date_to" class="form-label">Data Final</label>
                <input type="date" class="form-control" id="date_to" name="date_to" value="<?= e($dateTo) ?>">
            </div>
            <div class="col-md-2">
                <button type="submit" class="btn btn-primary w-100">
                    <i class="bi bi-search me-2"></i>Filtrar
                </button>
            </div>
            <div class="col-md-3">
                <a href="my_matches.php" class="btn btn-outline-secondary w-100">
                    <i class="bi bi-x-circle me-2"></i>Limpar
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Meus Personagens -->
<?php if (!empty($characters)): ?>
<div class="card mb-4">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="bi bi-person-lines-fill me-2"></i>Meus Personagens
        </h5>
    </div>
    <div class="card-body">
        <div class="row g-3">
            <?php foreach ($characters as $char): ?>
                <?php $charStats = getPlayerStats($char['char_id']); ?>
                <div class="col-md-4">
                    <a href="player.php?id=<?= $char['char_id'] ?>" class="text-decoration-none">
                        <div class="card h-100">
                            <div class="card-body">
                                <div class="d-flex align-items-center gap-3">
                                    <div class="player-avatar" style="width: 60px; height: 60px;">
                                        <img src="<?= classImage($char['class']) ?>" 
                                             alt="<?= e(getClassName($char['class'])) ?>"
                                             style="width: 100%; height: 100%;"
                                             onerror="this.src='assets/images/placeholder.png'">
                                    </div>
                                    <div class="flex-grow-1">
                                        <h6 class="mb-1"><?= e($char['name']) ?></h6>
                                        <small class="text-muted">
                                            <?= e(getClassName($char['class'])) ?> Lv.<?= $char['base_level'] ?>
                                        </small>
                                        <?php if ($charStats): ?>
                                            <div class="mt-1">
                                                <span class="badge bg-primary"><?= $charStats['rating_points'] ?> pts</span>
                                                <span class="badge bg-success"><?= $charStats['wins'] ?>W</span>
                                                <span class="badge bg-danger"><?= $charStats['losses'] ?>L</span>
                                            </div>
                                        <?php else: ?>
                                            <div class="mt-1">
                                                <span class="badge bg-secondary">Sem partidas</span>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </a>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Lista de Partidas -->
<?php if (empty($matches)): ?>
    <div class="card">
        <div class="card-body text-center py-5">
            <i class="bi bi-emoji-frown fs-1 d-block mb-3 text-muted"></i>
            <h5 class="text-muted">Nenhuma partida encontrada</h5>
            <p class="text-muted mb-0">
                <?php if (empty($characters)): ?>
                    Voce nao possui personagens registrados.
                <?php elseif (!empty($filters)): ?>
                    Tente ajustar os filtros de busca.
                <?php else: ?>
                    Participe de partidas na Arena 7x7 para ve-las aqui!
                <?php endif; ?>
            </p>
        </div>
    </div>
<?php else: ?>
    <div class="card">
        <div class="table-responsive">
            <table class="table table-striped mb-0">
                <thead>
                    <tr>
                        <th>Data</th>
                        <th>Personagem</th>
                        <th class="text-center">Time</th>
                        <th class="text-center">Placar</th>
                        <th class="text-center">K/D/A</th>
                        <th class="text-center">Resultado</th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($matches as $match): ?>
                        <?php 
                            $won = ($match['my_team'] == $match['winner_team']);
                            $draw = ($match['winner_team'] == 0);
                        ?>
                        <tr>
                            <td>
                                <small><?= formatDateTime($match['start_time']) ?></small>
                            </td>
                            <td>
                                <div class="player-info">
                                    <div class="player-avatar" style="width: 32px; height: 32px;">
                                        <img src="<?= classImage($match['my_class']) ?>" 
                                             alt="<?= e(getClassName($match['my_class'])) ?>"
                                             onerror="this.src='assets/images/placeholder.png'">
                                    </div>
                                    <div>
                                        <span class="player-name"><?= e($match['my_char_name']) ?></span>
                                    </div>
                                </div>
                            </td>
                            <td class="text-center">
                                <?php if ($match['my_team'] == 1): ?>
                                    <span class="team-badge team-badge-blue">Azul</span>
                                <?php else: ?>
                                    <span class="team-badge team-badge-red">Vermelho</span>
                                <?php endif; ?>
                            </td>
                            <td class="text-center">
                                <span class="text-primary fw-bold"><?= $match['team_blue_score'] ?></span>
                                <span class="text-muted">-</span>
                                <span class="text-danger fw-bold"><?= $match['team_red_score'] ?></span>
                            </td>
                            <td class="text-center">
                                <span class="badge badge-kills"><?= $match['kills'] ?></span>
                                <span class="badge badge-deaths"><?= $match['deaths'] ?></span>
                                <span class="badge badge-assists"><?= $match['assists'] ?></span>
                            </td>
                            <td class="text-center">
                                <?php if ($draw): ?>
                                    <span class="match-result match-result-draw">EMPATE</span>
                                <?php elseif ($won): ?>
                                    <span class="match-result match-result-victory">VITORIA</span>
                                <?php else: ?>
                                    <span class="match-result match-result-defeat">DERROTA</span>
                                <?php endif; ?>
                            </td>
                            <td class="text-end">
                                <a href="<?= matchUrl($match['match_id']) ?>" class="btn btn-sm btn-outline-primary">
                                    <i class="bi bi-eye"></i>
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Paginacao -->
    <?php if ($totalPages > 1): ?>
        <?php 
            $queryParams = http_build_query(array_filter([
                'char_id' => $selectedChar,
                'date_from' => $dateFrom,
                'date_to' => $dateTo
            ]));
        ?>
        <div class="mt-4">
            <?= paginate($page, $totalPages, 'my_matches.php?' . $queryParams) ?>
        </div>
    <?php endif; ?>
<?php endif; ?>

<?php include 'includes/footer.php'; ?>
