<?php
/**
 * Arena 7x7 - Perfil do Jogador
 */

define('ARENA7X7', true);
require_once 'includes/config.php';
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

Auth::startSession();

// Aceitar nome ou ID (prioriza nome)
$charName = getParam('name');
$charId = getInt('id');

if ($charName) {
    // Buscar por nome
    $charId = getCharIdByName($charName);
}

if (!$charId) {
    setFlash('error', 'Jogador nao encontrado.');
    redirect('ranking.php');
}

// Buscar estatisticas do jogador
$player = getPlayerStats($charId);

if (!$player) {
    setFlash('error', 'Jogador nao encontrado ou nao possui partidas.');
    redirect('ranking.php');
}

$pageTitle = $player['char_name'];

// Calcular estatisticas
$kda = calculateKDA($player['total_kills'], $player['total_deaths'], $player['total_assists']);
$winRate = calculateWinRate($player['wins'], $player['matches_played']);
$avgDamage = $player['matches_played'] > 0 ? round($player['total_damage'] / $player['matches_played']) : 0;
$avgHealing = $player['matches_played'] > 0 ? round($player['total_healing'] / $player['matches_played']) : 0;

// Buscar posicao no ranking
$rankSql = "SELECT COUNT(*) + 1 FROM arena7x7_ranking 
            WHERE points > :rating OR (points = :rating2 AND wins > :wins)";
$rankPosition = db()->fetchColumn($rankSql, [
    'rating' => $player['rating_points'],
    'rating2' => $player['rating_points'],
    'wins' => $player['wins']
]);

// Buscar partidas recentes do jogador
$recentMatches = getRecentMatches(5, 0, ['char_id' => $charId]);

// Buscar kills mais frequentes (vitimas)
$topVictims = db()->fetchAll(
    "SELECT c.name as victim_name, COUNT(*) as kill_count 
     FROM arena7x7_kills k
     JOIN `" . TABLE_CHAR . "` c ON c.char_id = k.victim_id
     WHERE k.killer_id = :char_id
     GROUP BY k.victim_id
     ORDER BY kill_count DESC
     LIMIT 5",
    ['char_id' => $charId]
);

// Buscar quem mais matou este jogador
$topKillers = db()->fetchAll(
    "SELECT c.name as killer_name, COUNT(*) as death_count 
     FROM arena7x7_kills k
     JOIN `" . TABLE_CHAR . "` c ON c.char_id = k.killer_id
     WHERE k.victim_id = :char_id
     GROUP BY k.killer_id
     ORDER BY death_count DESC
     LIMIT 5",
    ['char_id' => $charId]
);

// Buscar skills mais usadas
$topSkills = db()->fetchAll(
    "SELECT skill_id, skill_name, SUM(use_count) as total_uses, SUM(total_damage) as total_damage, SUM(kills_with_skill) as total_kills
     FROM arena7x7_skill_summary
     WHERE char_id = :char_id
     GROUP BY skill_id, skill_name
     ORDER BY total_damage DESC
     LIMIT 10",
    ['char_id' => $charId]
);

// Se não houver dados na nova tabela, tentar a antiga
if (empty($topSkills)) {
    $topSkills = db()->fetchAll(
        "SELECT skill_id, skill_name, SUM(hit_count) as total_uses, SUM(total_damage) as total_damage
         FROM arena7x7_damage_by_skill
         WHERE char_id = :char_id
         GROUP BY skill_id, skill_name
         ORDER BY total_damage DESC
         LIMIT 10",
        ['char_id' => $charId]
    );
}

// Buscar dano causado por alvo (quem mais apanhou deste jogador)
$topDamageTargets = db()->fetchAll(
    "SELECT 
        COALESCE(NULLIF(dt.target_name, ''), c.name) as target_name,
        SUM(dt.total_damage) as total_damage, 
        SUM(dt.skill_damage) as skill_damage, 
        SUM(dt.normal_damage) as normal_damage
     FROM arena7x7_damage_by_target dt
     LEFT JOIN `" . TABLE_CHAR . "` c ON c.char_id = dt.target_id
     WHERE dt.attacker_id = :char_id
     GROUP BY dt.target_id, target_name
     ORDER BY total_damage DESC
     LIMIT 5",
    ['char_id' => $charId]
);

// Buscar dano recebido por atacante (quem mais bateu neste jogador)
$topDamageReceivedFrom = db()->fetchAll(
    "SELECT 
        COALESCE(NULLIF(dt.attacker_name, ''), c.name) as attacker_name,
        SUM(dt.total_damage) as total_damage
     FROM arena7x7_damage_by_target dt
     LEFT JOIN `" . TABLE_CHAR . "` c ON c.char_id = dt.attacker_id
     WHERE dt.target_id = :char_id
     GROUP BY dt.attacker_id, attacker_name
     ORDER BY total_damage DESC
     LIMIT 5",
    ['char_id' => $charId]
);

include 'includes/header.php';
?>

<!-- Cabecalho do Perfil -->
<div class="card mb-4">
    <div class="card-body">
        <div class="row align-items-center">
            <div class="col-md-2 text-center">
                <div class="player-avatar mx-auto" style="width: 100px; height: 100px;">
                    <img src="<?= classImage($player['class']) ?>" 
                         alt="<?= e(getClassName($player['class'])) ?>"
                         style="width: 100%; height: 100%;"
                         onerror="this.src='assets/images/placeholder.png'">
                </div>
            </div>
            <div class="col-md-6">
                <h1 class="mb-1"><?= e($player['char_name']) ?></h1>
                <p class="text-muted mb-2">
                    <?= e(getClassName($player['class'])) ?> - Nivel <?= $player['base_level'] ?>
                </p>
                <div class="d-flex gap-3">
                    <span class="badge bg-warning fs-6">
                        <i class="bi bi-trophy me-1"></i>Rank #<?= $rankPosition ?>
                    </span>
                    <span class="badge bg-primary fs-6">
                        <i class="bi bi-star me-1"></i><?= formatNumber($player['rating_points']) ?> pts
                    </span>
                </div>
            </div>
            <div class="col-md-4 text-md-end mt-3 mt-md-0">
                <a href="ranking.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-2"></i>Voltar ao Ranking
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Estatisticas Principais -->
<div class="row g-4 mb-4">
    <div class="col-md-2 col-6">
        <div class="card stat-card stat-rating">
            <i class="bi bi-controller stat-icon"></i>
            <div class="stat-value"><?= $player['matches_played'] ?></div>
            <div class="stat-label">Partidas</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card stat-wins">
            <i class="bi bi-trophy stat-icon"></i>
            <div class="stat-value"><?= $player['wins'] ?></div>
            <div class="stat-label">Vitorias</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card stat-deaths">
            <i class="bi bi-x-circle stat-icon"></i>
            <div class="stat-value"><?= $player['losses'] ?></div>
            <div class="stat-label">Derrotas</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card">
            <i class="bi bi-percent stat-icon text-info"></i>
            <div class="stat-value <?= getWinRateClass($winRate) ?>"><?= $winRate ?>%</div>
            <div class="stat-label">Win Rate</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card">
            <i class="bi bi-graph-up stat-icon text-success"></i>
            <div class="stat-value <?= getKdaClass($kda) ?>"><?= number_format($kda, 2) ?></div>
            <div class="stat-label">KDA</div>
        </div>
    </div>
    <div class="col-md-2 col-6">
        <div class="card stat-card stat-kills">
            <i class="bi bi-lightning stat-icon"></i>
            <div class="stat-value"><?= $player['total_kills'] ?></div>
            <div class="stat-label">Kills</div>
        </div>
    </div>
</div>

<!-- Estatisticas Detalhadas -->
<div class="row g-4 mb-4">
    <div class="col-md-3 col-6">
        <div class="card h-100">
            <div class="card-body text-center">
                <i class="bi bi-crosshair fs-1 text-danger mb-2"></i>
                <h3 class="mb-1"><?= formatNumber($player['total_damage']) ?></h3>
                <p class="text-muted mb-0">Dano Total</p>
                <small class="text-muted">Media: <?= formatNumber($avgDamage) ?>/partida</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 col-6">
        <div class="card h-100">
            <div class="card-body text-center">
                <i class="bi bi-heart-pulse fs-1 text-success mb-2"></i>
                <h3 class="mb-1"><?= formatNumber($player['total_healing']) ?></h3>
                <p class="text-muted mb-0">Cura Total</p>
                <small class="text-muted">Media: <?= formatNumber($avgHealing) ?>/partida</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 col-6">
        <div class="card h-100">
            <div class="card-body text-center">
                <i class="bi bi-people fs-1 text-info mb-2"></i>
                <h3 class="mb-1"><?= $player['total_assists'] ?></h3>
                <p class="text-muted mb-0">Assists</p>
            </div>
        </div>
    </div>
    <div class="col-md-3 col-6">
        <div class="card h-100">
            <div class="card-body text-center">
                <i class="bi bi-emoji-frown fs-1 text-secondary mb-2"></i>
                <h3 class="mb-1"><?= $player['total_deaths'] ?></h3>
                <p class="text-muted mb-0">Mortes</p>
            </div>
        </div>
    </div>
</div>

<div class="row g-4">
    <!-- Skills Mais Usadas -->
    <div class="col-lg-6">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-magic me-2 text-primary"></i>Skills Mais Usadas
                </h5>
            </div>
            <div class="card-body">
                <?php if (empty($topSkills)): ?>
                    <p class="text-center text-muted py-4">Nenhuma skill registrada ainda.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-sm mb-0">
                            <thead>
                                <tr>
                                    <th>Skill</th>
                                    <th class="text-center">Usos</th>
                                    <th class="text-center">Kills</th>
                                    <th class="text-end">Dano</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($topSkills as $skill): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center gap-2">
                                                <img src="<?= skillImage($skill['skill_id']) ?>" 
                                                     alt="<?= e($skill['skill_name'] ?? 'Skill') ?>"
                                                     class="skill-icon"
                                                     onerror="this.style.display='none'">
                                                <span><?= e($skill['skill_name'] ?? 'Skill #' . $skill['skill_id']) ?></span>
                                            </div>
                                        </td>
                                        <td class="text-center"><?= $skill['total_uses'] ?></td>
                                        <td class="text-center">
                                            <?php if (!empty($skill['total_kills'])): ?>
                                                <span class="badge bg-danger"><?= $skill['total_kills'] ?></span>
                                            <?php else: ?>
                                                <span class="text-muted">0</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="text-end text-danger"><?= formatNumber($skill['total_damage']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Damage Log -->
    <div class="col-lg-6">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-crosshair me-2 text-danger"></i>Damage Log
                </h5>
            </div>
            <div class="card-body">
                <?php if (empty($topDamageTargets)): ?>
                    <p class="text-center text-muted py-4">Nenhum dano registrado ainda.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-sm mb-0">
                            <thead>
                                <tr>
                                    <th>Alvo</th>
                                    <th class="text-end">Skills</th>
                                    <th class="text-end">Normal</th>
                                    <th class="text-end">Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($topDamageTargets as $target): ?>
                                    <tr>
                                        <td>
                                            <a href="<?= playerUrl($target['target_name']) ?>" class="player-name">
                                                <?= e($target['target_name']) ?>
                                            </a>
                                        </td>
                                        <td class="text-end text-info"><?= formatNumber($target['skill_damage']) ?></td>
                                        <td class="text-end text-warning"><?= formatNumber($target['normal_damage']) ?></td>
                                        <td class="text-end text-danger fw-bold"><?= formatNumber($target['total_damage']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Rivalidades -->
<div class="row g-4 mt-2">
    <!-- Dano Recebido de -->
    <div class="col-lg-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-shield-exclamation me-2 text-warning"></i>Mais Dano Recebido De
                </h5>
            </div>
            <div class="card-body py-2">
                <?php if (empty($topDamageReceivedFrom)): ?>
                    <p class="text-center text-muted py-3">Nenhum dano recebido registrado.</p>
                <?php else: ?>
                    <?php foreach ($topDamageReceivedFrom as $attacker): ?>
                        <div class="d-flex justify-content-between align-items-center py-2 border-bottom">
                            <a href="<?= playerUrl($attacker['attacker_name']) ?>" class="player-name">
                                <?= e($attacker['attacker_name']) ?>
                            </a>
                            <span class="text-danger fw-bold"><?= formatNumber($attacker['total_damage']) ?></span>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Top Vitimas -->
    <div class="col-lg-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-crosshair me-2 text-danger"></i>Vitimas Favoritas
                </h5>
            </div>
            <div class="card-body py-2">
                <?php if (empty($topVictims)): ?>
                    <p class="text-center text-muted py-3">Nenhuma kill ainda.</p>
                <?php else: ?>
                    <?php foreach ($topVictims as $victim): ?>
                        <div class="d-flex justify-content-between align-items-center py-2 border-bottom">
                            <span><?= e($victim['victim_name']) ?></span>
                            <span class="badge bg-danger"><?= $victim['kill_count'] ?> kills</span>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Top Killers -->
    <div class="col-lg-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-emoji-angry me-2 text-secondary"></i>Maiores Nemesis
                </h5>
            </div>
            <div class="card-body py-2">
                <?php if (empty($topKillers)): ?>
                    <p class="text-center text-muted py-3">Nenhuma morte ainda.</p>
                <?php else: ?>
                    <?php foreach ($topKillers as $killer): ?>
                        <div class="d-flex justify-content-between align-items-center py-2 border-bottom">
                            <span><?= e($killer['killer_name']) ?></span>
                            <span class="badge bg-secondary"><?= $killer['death_count'] ?> mortes</span>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Partidas Recentes -->
<div class="card mt-4">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="card-title mb-0">
            <i class="bi bi-clock-history me-2"></i>Partidas Recentes
        </h5>
        <a href="matches.php?player=<?= urlencode($player['char_name']) ?>" class="btn btn-sm btn-outline-primary">
            Ver todas
        </a>
    </div>
    <div class="card-body">
        <?php if (empty($recentMatches)): ?>
            <p class="text-center text-secondary py-4">Nenhuma partida encontrada.</p>
        <?php else: ?>
            <div class="row g-3">
                <?php foreach ($recentMatches as $match): ?>
                    <?php 
                        // Verificar se o jogador ganhou ou perdeu
                        $playerTeam = db()->fetchColumn(
                            "SELECT team FROM arena7x7_match_players WHERE match_id = :match_id AND char_id = :char_id",
                            ['match_id' => $match['match_id'], 'char_id' => $charId]
                        );
                        $won = ($playerTeam == $match['winner_team']);
                    ?>
                    <div class="col-md-4">
                        <a href="<?= matchUrl($match['match_id']) ?>" class="text-decoration-none">
                            <div class="card h-100 <?= $won ? 'border-success' : ($match['winner_team'] == 0 ? '' : 'border-danger') ?>">
                                <div class="card-body text-center">
                                    <small class="text-secondary d-block mb-2">
                                        <?= formatDateTime($match['start_time']) ?>
                                    </small>
                                    <div class="fs-4 fw-bold">
                                        <span class="text-primary"><?= $match['team_blue_score'] ?></span>
                                        <span class="text-secondary mx-2">-</span>
                                        <span class="text-danger"><?= $match['team_red_score'] ?></span>
                                    </div>
                                    <?php if ($won): ?>
                                        <span class="badge bg-success mt-2">VITORIA</span>
                                    <?php elseif ($match['winner_team'] == 0): ?>
                                        <span class="badge bg-secondary mt-2">EMPATE</span>
                                    <?php else: ?>
                                        <span class="badge bg-danger mt-2">DERROTA</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </a>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include 'includes/footer.php'; ?>
