<?php
/**
 * Arena 7x7 - Busca Avancada
 */

define('ARENA7X7', true);
require_once 'includes/config.php';
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

Auth::startSession();

$pageTitle = 'Buscar';

// Parametros de busca
$searchType = getParam('type', 'player');
$searchQuery = getParam('q');
$dateFrom = getParam('date_from');
$dateTo = getParam('date_to');

$results = [];
$searched = false;

if (!empty($searchQuery) || !empty($dateFrom) || !empty($dateTo)) {
    $searched = true;
    
    if ($searchType === 'player') {
        // Buscar jogadores por nome
        $sql = "SELECT 
                    r.*,
                    r.char_name,
                    r.points as rating_points,
                    r.total_damage_done as total_damage,
                    r.total_healing_done as total_healing,
                    c.class,
                    c.base_level
                FROM arena7x7_ranking r
                LEFT JOIN `" . TABLE_CHAR . "` c ON c.char_id = r.char_id
                WHERE r.char_name LIKE :search
                ORDER BY r.points DESC
                LIMIT 50";
        
        $results = db()->fetchAll($sql, ['search' => '%' . $searchQuery . '%']);
        
    } elseif ($searchType === 'match') {
        // Buscar partidas por ID ou data
        $filters = [];
        if (!empty($searchQuery) && is_numeric($searchQuery)) {
            $filters['match_id'] = (int)$searchQuery;
        }
        if ($dateFrom) $filters['date_from'] = $dateFrom;
        if ($dateTo) $filters['date_to'] = $dateTo;
        
        if (!empty($filters['match_id'])) {
            $match = getMatchDetails($filters['match_id']);
            if ($match) {
                $results = [$match];
            }
        } else {
            $results = getRecentMatches(50, 0, $filters);
        }
        
    } elseif ($searchType === 'opponent') {
        // Buscar partidas contra um jogador especifico
        $sql = "SELECT c.char_id FROM `" . TABLE_CHAR . "` c WHERE c.name LIKE :search LIMIT 1";
        $opponent = db()->fetchOne($sql, ['search' => '%' . $searchQuery . '%']);
        
        if ($opponent) {
            $sql = "SELECT DISTINCT m.* 
                    FROM arena7x7_matches m
                    JOIN arena7x7_match_players mp ON mp.match_id = m.match_id
                    WHERE mp.char_id = :char_id
                    ORDER BY m.start_time DESC
                    LIMIT 50";
            $results = db()->fetchAll($sql, ['char_id' => $opponent['char_id']]);
        }
    }
}

include 'includes/header.php';
?>

<div class="mb-4">
    <h1 class="mb-1"><i class="bi bi-search me-2 text-primary"></i>Buscar</h1>
    <p class="text-muted mb-0">Pesquise por jogadores, partidas ou adversarios</p>
</div>

<!-- Formulario de Busca -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" action="">
            <div class="row g-3">
                <div class="col-md-3">
                    <label for="type" class="form-label">Tipo de Busca</label>
                    <select class="form-select" id="type" name="type">
                        <option value="player" <?= $searchType === 'player' ? 'selected' : '' ?>>
                            Jogador (por nome)
                        </option>
                        <option value="match" <?= $searchType === 'match' ? 'selected' : '' ?>>
                            Partida (por ID ou data)
                        </option>
                        <option value="opponent" <?= $searchType === 'opponent' ? 'selected' : '' ?>>
                            Partidas contra (nome do adversario)
                        </option>
                    </select>
                </div>
                <div class="col-md-5">
                    <label for="q" class="form-label">Buscar</label>
                    <div class="input-group">
                        <span class="input-group-text bg-transparent">
                            <i class="bi bi-search"></i>
                        </span>
                        <input type="text" 
                               class="form-control" 
                               id="q" 
                               name="q" 
                               value="<?= e($searchQuery) ?>"
                               placeholder="Nome do jogador ou ID da partida...">
                    </div>
                </div>
                <div class="col-md-2">
                    <label for="date_from" class="form-label">Data Inicial</label>
                    <input type="date" class="form-control" id="date_from" name="date_from" value="<?= e($dateFrom) ?>">
                </div>
                <div class="col-md-2">
                    <label for="date_to" class="form-label">Data Final</label>
                    <input type="date" class="form-control" id="date_to" name="date_to" value="<?= e($dateTo) ?>">
                </div>
            </div>
            <div class="row mt-3">
                <div class="col-12">
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-search me-2"></i>Buscar
                    </button>
                    <a href="search.php" class="btn btn-outline-secondary ms-2">
                        <i class="bi bi-x-circle me-2"></i>Limpar
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Resultados -->
<?php if ($searched): ?>
    <div class="card">
        <div class="card-header">
            <h5 class="card-title mb-0">
                <i class="bi bi-list-ul me-2"></i>
                Resultados 
                <span class="badge bg-primary ms-2"><?= count($results) ?></span>
            </h5>
        </div>
        
        <?php if (empty($results)): ?>
            <div class="card-body text-center py-5">
                <i class="bi bi-emoji-frown fs-1 d-block mb-3 text-muted"></i>
                <h5 class="text-muted">Nenhum resultado encontrado</h5>
                <p class="text-muted mb-0">Tente termos de busca diferentes.</p>
            </div>
        <?php elseif ($searchType === 'player'): ?>
            <!-- Resultados de Jogadores -->
            <div class="table-responsive">
                <table class="table table-striped mb-0">
                    <thead>
                        <tr>
                            <th>Jogador</th>
                            <th class="text-center">Rating</th>
                            <th class="text-center">Partidas</th>
                            <th class="text-center">V/D</th>
                            <th class="text-center">Win Rate</th>
                            <th class="text-center">KDA</th>
                            <th></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($results as $player): ?>
                            <?php 
                                $kda = calculateKDA($player['total_kills'], $player['total_deaths'], $player['total_assists']);
                                $winRate = calculateWinRate($player['wins'], $player['matches_played']);
                            ?>
                            <tr>
                                <td>
                                    <div class="player-info">
                                        <div class="player-avatar">
                                            <img src="<?= classImage($player['class']) ?>" 
                                                 alt="<?= e(getClassName($player['class'])) ?>"
                                                 onerror="this.src='assets/images/placeholder.png'">
                                        </div>
                                        <div>
                                            <span class="player-name"><?= e($player['char_name']) ?></span>
                                            <div class="player-class">
                                                <?= e(getClassName($player['class'])) ?> Lv.<?= $player['base_level'] ?>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <span class="text-warning fw-bold"><?= formatNumber($player['rating_points']) ?></span>
                                </td>
                                <td class="text-center"><?= $player['matches_played'] ?></td>
                                <td class="text-center">
                                    <span class="text-success"><?= $player['wins'] ?></span>
                                    /
                                    <span class="text-danger"><?= $player['losses'] ?></span>
                                </td>
                                <td class="text-center">
                                    <span class="<?= getWinRateClass($winRate) ?>"><?= $winRate ?>%</span>
                                </td>
                                <td class="text-center">
                                    <span class="<?= getKdaClass($kda) ?>"><?= number_format($kda, 2) ?></span>
                                </td>
                                <td class="text-end">
                                    <a href="<?= playerUrl($player['char_name']) ?>" class="btn btn-sm btn-primary">
                                        <i class="bi bi-eye me-1"></i>Ver Perfil
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <!-- Resultados de Partidas -->
            <div class="card-body">
                <div class="row g-4">
                    <?php foreach ($results as $match): ?>
                        <div class="col-md-6">
                            <a href="<?= matchUrl($match['match_id']) ?>" class="text-decoration-none">
                                <div class="card match-card h-100">
                                    <div class="match-header">
                                        <span class="match-id">
                                            <i class="bi bi-controller me-1"></i>Partida
                                        </span>
                                        <span class="match-date">
                                            <i class="bi bi-calendar me-1"></i>
                                            <?= formatDateTime($match['start_time']) ?>
                                        </span>
                                    </div>
                                    
                                    <div class="match-score">
                                        <div class="team-score team-blue">
                                            <?= $match['team_blue_score'] ?>
                                        </div>
                                        <div class="vs">VS</div>
                                        <div class="team-score team-red">
                                            <?= $match['team_red_score'] ?>
                                        </div>
                                    </div>
                                    
                                    <div class="text-center mt-2">
                                        <?php if ($match['winner_team'] == 1): ?>
                                            <span class="team-badge team-badge-blue"><?= e($match['blue_guild_name']) ?> Venceu</span>
                                        <?php elseif ($match['winner_team'] == 2): ?>
                                            <span class="team-badge team-badge-red"><?= e($match['red_guild_name']) ?> Venceu</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Empate</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </a>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
<?php else: ?>
    <!-- Instrucoes -->
    <div class="card">
        <div class="card-body text-center py-5">
            <i class="bi bi-lightbulb fs-1 d-block mb-3 text-warning"></i>
            <h5>Dicas de Busca</h5>
            <div class="text-muted">
                <p><strong>Jogador:</strong> Digite parte do nome do personagem</p>
                <p><strong>Partida:</strong> Digite o ID da partida ou selecione um intervalo de datas</p>
                <p><strong>Partidas contra:</strong> Digite o nome do adversario para ver todas as partidas em que ele participou</p>
            </div>
        </div>
    </div>
<?php endif; ?>

<?php include 'includes/footer.php'; ?>
